/******/ (() => { // webpackBootstrap
/**
 * 支付宝页面检测脚本
 * 用于检测和处理支付宝H5支付页面
 */

(function () {
  'use strict';

  console.log('支付宝页面检测脚本已加载，当前URL:', window.location.href);

  // 检查当前页面是否为支付宝支付页面
  function isAlipayPaymentPage() {
    var url = window.location.href;
    var hostname = window.location.hostname;
    var pathname = window.location.pathname;
    var title = document.title;
    console.log('检测支付宝页面 - URL:', url, 'Title:', title);
    console.log('检测支付宝页面 - Hostname:', hostname, 'Pathname:', pathname);

    // 检查URL - 使用hostname和pathname精确匹配,避免referer干扰
    if (hostname === 'mclient.alipay.com' && pathname.includes('/h5pay/unifiedLogin/')) {
      console.log('通过hostname和pathname检测到支付宝登录页面');
      return true;
    }

    // 检查标题
    if (title && (title.includes('支付宝') || title.includes('alipay') || title.includes('统一收银台') || title.includes('登录'))) {
      console.log('通过标题检测到支付宝页面');
      return true;
    }

    // 检查页面内容（添加安全检查）
    try {
      var bodyText = document && document.body ? document.body.innerText || '' : '';
      if (bodyText.includes('支付宝') || bodyText.includes('扫码支付') || bodyText.includes('确认付款') || bodyText.includes('请登录支付宝账户')) {
        console.log('通过页面内容检测到支付宝页面');
        return true;
      }
    } catch (error) {
      console.log('Body text check error in alipayDetector:', error.message);
      // 继续执行，不影响其他检测逻辑
    }

    // 检查特定元素（添加安全检查）
    try {
      if (document && typeof document.querySelector === 'function') {
        if (document.querySelector('[class*="alipay"]') || document.querySelector('[id*="alipay"]') || document.querySelector('[class*="ant-"]')) {
          console.log('通过页面元素检测到支付宝页面');
          return true;
        }
      }
    } catch (error) {
      console.log('querySelector error in alipayDetector:', error.message);
      // 继续执行，不影响其他检测逻辑
    }
    return false;
  }

  // 发送检测结果给父窗口
  function sendDetectionResult() {
    var detectionData = {
      type: 'ALIPAY_PAGE_DETECTED',
      from: 'alipay_page',
      url: window.location.href,
      title: document.title,
      timestamp: Date.now()
    };
    console.log('发送检测结果:', detectionData);

    // 尝试向opener发送消息
    if (window.opener && !window.opener.closed) {
      try {
        window.opener.postMessage(detectionData, '*');
        console.log('已向opener发送消息');
      } catch (e) {
        console.log('无法向opener发送消息:', e.message);
      }
    }

    // 也尝试向parent发送消息
    if (window.parent && window.parent !== window) {
      try {
        window.parent.postMessage(detectionData, '*');
        console.log('已向parent发送消息');
      } catch (e) {
        console.log('无法向parent发送消息:', e.message);
      }
    }

    // 向所有窗口广播（Chrome扩展环境下的备用方案）
    try {
      window.postMessage(detectionData, '*');
      console.log('已广播消息');
    } catch (e) {
      console.log('无法广播消息:', e.message);
    }
  }

  // 监听来自父窗口的消息
  window.addEventListener('message', function (event) {
    if (event.data && event.data.type === 'CHECK_ALIPAY_PAGE' && event.data.from === 'pdd_extension') {
      console.log('收到检测请求，检查是否为支付宝页面');
      if (isAlipayPaymentPage()) {
        console.log('确认为支付宝支付页面，发送确认消息');
        sendDetectionResult();
      } else {
        console.log('当前页面不是支付宝支付页面');
      }
    }
  });

  // 页面加载完成后自动检测
  function checkAndReport() {
    console.log('自动检测支付宝页面...');
    if (isAlipayPaymentPage()) {
      console.log('自动检测到支付宝支付页面，发送通知');
      sendDetectionResult();
    } else {
      console.log('自动检测：当前页面不是支付宝支付页面');
    }
  }

  // 多次检测，确保不会错过
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function () {
      checkAndReport();
      // 大幅延长检测时间，增加检测次数
      setTimeout(checkAndReport, 1000);
      setTimeout(checkAndReport, 2000);
      setTimeout(checkAndReport, 3000);
      setTimeout(checkAndReport, 5000);
      setTimeout(checkAndReport, 8000);
      setTimeout(checkAndReport, 10000);
      setTimeout(checkAndReport, 15000);
      setTimeout(checkAndReport, 20000);
    });
  } else {
    checkAndReport();
    setTimeout(checkAndReport, 1000);
    setTimeout(checkAndReport, 2000);
    setTimeout(checkAndReport, 3000);
    setTimeout(checkAndReport, 5000);
    setTimeout(checkAndReport, 8000);
    setTimeout(checkAndReport, 10000);
    setTimeout(checkAndReport, 15000);
    setTimeout(checkAndReport, 20000);
  }

  // 监听URL变化（对于单页应用）
  var currentUrl = window.location.href;
  var observer = new MutationObserver(function (mutations) {
    if (window.location.href !== currentUrl) {
      currentUrl = window.location.href;
      console.log('URL发生变化，重新检测:', currentUrl);
      // 大幅增加检测频率和时长，确保不错过unifiedLogin页面
      setTimeout(checkAndReport, 100);
      setTimeout(checkAndReport, 300);
      setTimeout(checkAndReport, 500);
      setTimeout(checkAndReport, 800);
      setTimeout(checkAndReport, 1000);
      setTimeout(checkAndReport, 1500);
      setTimeout(checkAndReport, 2000);
      setTimeout(checkAndReport, 3000);
      setTimeout(checkAndReport, 5000);
      setTimeout(checkAndReport, 8000);
      setTimeout(checkAndReport, 10000);
      setTimeout(checkAndReport, 15000);
    }

    // 监听DOM变化，可能有动态内容
    var hasAlipayContent = false;
    mutations.forEach(function (mutation) {
      mutation.addedNodes.forEach(function (node) {
        if (node.nodeType === 1) {
          // 元素节点
          var text = node.textContent || '';
          if (text.includes('支付宝') || text.includes('alipay') || text.includes('登录')) {
            hasAlipayContent = true;
          }
        }
      });
    });
    if (hasAlipayContent) {
      console.log('检测到新增的支付宝相关内容');
      setTimeout(checkAndReport, 1000);
    }
  });
  if (document.body) {
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  } else {
    // 如果body还没有加载，等待一会儿
    setTimeout(function () {
      if (document.body) {
        observer.observe(document.body, {
          childList: true,
          subtree: true
        });
      }
    }, 1000);
  }
  console.log('支付宝页面检测脚本初始化完成');

  // 备用URL检测：使用定时器定期检查URL变化
  setInterval(function () {
    if (window.location.href !== currentUrl) {
      currentUrl = window.location.href;
      console.log('定时器检测到URL变化:', currentUrl);
      checkAndReport();
    }
  }, 500); // 每500ms检查一次URL变化，提高检测频率
})();
/******/ })()
;